//
//  FileListViewController.swift
//  NeoSpectraMicroSwift
//
//  Created by Si-Ware on 1/30/18.
//  Copyright © 2018 siware. All rights reserved.
//

import Foundation

class FileListViewController: UIViewController {
    
    // TableView
    @IBOutlet weak var tableView: UITableView!
    let collation = UILocalizedIndexedCollation.current()
    
    /// Data
    var didSelectFile: ((FBFile) -> ())?
    var files = [FBFile]()
    var initialPath: URL?
    let parser = FileParser.sharedInstance
    let previewManager = PreviewManager()
    var sections: [[FBFile]] = []
    var allowEditing: Bool = false
    var save_load = ""
    var selectAll = false
    var selectedFiles = [URL]()
    var chartView : ChartViewController?
    var PSDView : PSDViewController?
    var SpectrumView : SpectrumViewController?
    
    
    // Search controller
    var filteredFiles = [FBFile]()
    let searchController: UISearchController = {
        let searchController = UISearchController(searchResultsController: nil)
        searchController.searchBar.searchBarStyle = .minimal
        searchController.searchBar.backgroundColor = UIColor.white
        searchController.dimsBackgroundDuringPresentation = false
        return searchController
    }()
    
    
    //MARK: Lifecycle
    convenience init (initialPath: URL, save_load: String, selectAll: Bool) {
        self.init(initialPath: initialPath, showCancelButton: true, save_load: save_load, selectAll: selectAll)
    }
    
    convenience init (initialPath: URL, showCancelButton: Bool, save_load: String, selectAll: Bool) {
        self.init(nibName: "FileBrowser", bundle: Bundle(for: FileListViewController.self))
        self.edgesForExtendedLayout = UIRectEdge()
        
        // Set initial path
        self.initialPath = initialPath
        self.title = initialPath.lastPathComponent
        
        // Set search controller delegates
        searchController.searchResultsUpdater = self
        searchController.searchBar.delegate = self
        searchController.delegate = self
        
        self.save_load = save_load
        self.selectAll = selectAll
        
        if showCancelButton {
            // Add dismiss button
            let dismissButton = UIBarButtonItem(barButtonSystemItem: .cancel, target: self, action: #selector(FileListViewController.dismiss(button:)))
            self.navigationItem.rightBarButtonItem = dismissButton
        }
        
        if(save_load == "save"){
            let saveButton = UIBarButtonItem(barButtonSystemItem: .save, target: self, action: #selector(FileListViewController.saveData))
            let spacer = UIBarButtonItem(barButtonSystemItem: .flexibleSpace, target: nil, action: nil)
            let createFolderButton = UIBarButtonItem(title: "Create Folder", style: .plain, target: self, action: #selector(FileListViewController.createFolder))
            //self.navigationItem.leftBarButtonItems = [saveButton, createFolderButton]
            var items = [UIBarButtonItem]()
            items.append(saveButton)
            items.append(spacer)
            items.append(createFolderButton)
            self.toolbarItems = items
        }else if(save_load == "load"){
            let loadButton = UIBarButtonItem(title: "Load", style: .plain, target: self, action: #selector(FileListViewController.loadData))
            let spacer = UIBarButtonItem(barButtonSystemItem: .flexibleSpace, target: nil, action: nil)
            let selectAllButton = UIBarButtonItem(title: "De/Select All", style: .plain, target: self, action: #selector(FileListViewController.selectAllFiles))
            //self.navigationItem.leftBarButtonItems = [loadButton, selectAllButton]
            var items = [UIBarButtonItem]()
            items.append(loadButton)
            items.append(spacer)
            items.append(selectAllButton)
            self.toolbarItems = items
        }
    }
    
    deinit{
        if #available(iOS 9.0, *) {
            searchController.loadViewIfNeeded()
        } else {
            searchController.loadView()
        }
    }
    
    func prepareData() {
        // Prepare data
        if let initialPath = initialPath {
            files = parser.filesForDirectory(initialPath)
            indexFiles()
        }
    }
    
    //MARK: UIViewController
    
    override func viewDidLoad() {
        
        prepareData()
        
        // Set search bar
        tableView.tableHeaderView = searchController.searchBar
        
        // Register for 3D touch
        self.registerFor3DTouch()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        // Scroll to hide search bar
        self.tableView.contentOffset = CGPoint(x: 0, y: searchController.searchBar.frame.size.height)
        
        // Make sure navigation bar is visible
        self.navigationController?.isNavigationBarHidden = false
        
        self.navigationController?.isToolbarHidden = false
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        self.selectAll = false
        if(self.selectedFiles.count > 0){
            self.selectedFiles.removeAll()
        }
        self.tableView.reloadData()
    }
    
    @objc func dismiss(button: UIBarButtonItem = UIBarButtonItem()) {
        self.dismiss(animated: true, completion: nil)
    }
    
    func selectAllFiles(){
        if(self.selectAll){
            self.selectAll = false
        }else{
            self.selectAll = true
        }
        self.tableView.reloadData()
    }
    
    func saveData(){
        self.chartView?.saveDataLogic(path: self.initialPath!)
        self.dismiss(animated: true, completion: nil)
    }
    
    func creatFolderLogic(name: String){
        let fileManager = FileManager.default
        let folderPath = self.initialPath?.appendingPathComponent(name)
        do{
            try fileManager.createDirectory(at: folderPath!, withIntermediateDirectories: true, attributes: nil)
        }
        catch{}
    }
    
    func showEmptyNameError(){
        let alertController = UIAlertController(title: "Folder Name Cannot be Empty", message:
            "", preferredStyle: UIAlertControllerStyle.alert)
        
        
        alertController.addAction(UIAlertAction(title: "Ok", style: UIAlertActionStyle.default,handler: { (UIAlertAction)in
            self.createFolder()
            
        }))
        
        DispatchQueue.main.async {
            self.present(alertController, animated: true, completion: nil)
        }
    }
    
    func createFolder(){
        let alertController = UIAlertController(title: "Create Folder", message:
            "Please enter folder name.", preferredStyle: UIAlertControllerStyle.alert)
        alertController.addTextField { (configurationTextField) in
            //configure here your textfield
            configurationTextField.textColor = UIColor.black
            configurationTextField.keyboardType = UIKeyboardType.namePhonePad
        }
        
        alertController.addAction(UIAlertAction(title: "Set", style: UIAlertActionStyle.default,handler: { (UIAlertAction)in
            let textField = (alertController.textFields?.first)! as UITextField
            if(textField.text == ""){
                self.showEmptyNameError()
            }else{
                self.creatFolderLogic(name: textField.text!)
                self.prepareData()
                self.tableView.reloadData()
            }
        }))
        
        alertController.addAction(UIAlertAction(title: "Cancel", style: UIAlertActionStyle.default, handler:nil))
        
        DispatchQueue.main.async {
            self.present(alertController, animated: true, completion: nil)
        }
    }
    
    func loadData(){
        if(self.PSDView != nil){
            self.PSDView?.loadDataLogic(dataPaths: self.selectedFiles)
        }
        
        if(self.SpectrumView != nil){
            self.SpectrumView?.loadDataLogic(dataPaths: self.selectedFiles)
        }
        self.dismiss(animated: true, completion: nil)
    }
    
    //MARK: Data
    
    func indexFiles() {
        let selector: Selector = #selector(getter: FBFile.displayName)
        sections = Array(repeating: [], count: collation.sectionTitles.count)
        if let sortedObjects = collation.sortedArray(from: files, collationStringSelector: selector) as? [FBFile]{
            for object in sortedObjects {
                let sectionNumber = collation.section(for: object, collationStringSelector: selector)
                sections[sectionNumber].append(object)
            }
        }
    }
    
    func fileForIndexPath(_ indexPath: IndexPath) -> FBFile {
        var file: FBFile
        if searchController.isActive {
            file = filteredFiles[(indexPath as NSIndexPath).row]
        }
        else {
            file = sections[(indexPath as NSIndexPath).section][(indexPath as NSIndexPath).row]
        }
        return file
    }
    
    func filterContentForSearchText(_ searchText: String) {
        filteredFiles = files.filter({ (file: FBFile) -> Bool in
            return file.displayName.lowercased().contains(searchText.lowercased())
        })
        tableView.reloadData()
    }
    
}
