﻿/*Spider GUI v0 by Nireos
 * 
 * Spider system user interface
 * 
 * 
 * A.R. 30/06/21
 */
using System;
using System.Drawing;
using System.Windows.Forms;
using SpiderLibrary;
using System.IO;
using System.IO.Ports;
using System.Globalization;
using System.Collections.Generic;

namespace SpiderGUI
{
    public partial class FormSpider : Form
    {
        public const int MAX_LENGTH = 256;  //number of points received for each packet from Silicon/InGaAs channel
        public SpiderClass Spider1;         //SpiderClass definition
        byte connectionStatus = 0;          //Connection flag: 0=not connected, 1=connected

        double chart_max = 10;              //chart maximum value
        double chart_min = -10;             //chart minimum value

        byte time_to_update = 0;            //timer flag to limit GUI refresh rate
        double[] plot_voltage_Si_array = new double[MAX_LENGTH];        //array of 256 elements containing the voltage values from Silicon (Si) channel for each packet 
        double[] plot_voltage_InGaAs_array = new double[MAX_LENGTH];    //array of 256 elements containing the voltage values from InGaAs channel for each packet 

        double[] time_ms_array = new double[MAX_LENGTH];        //array of 256 elements containing the time values in ms
        uint[] sampling_rate_combo = new uint[11] { 500, 1000, 2500, 5000, 10000, 16000, 32000, 64000, 80000, 100000, 120000};  //array containing the sampling rate numerical values

        string save_directory = @"";        //main saving folder
        string save_path = "";              //save path including the time-data 
        Queue<string> stream_data = new Queue<string>();    //queue where data is temperarily stored when recording data stream

        public FormSpider()     //form constructor
        {
            InitializeComponent();

            Spider1 = new SpiderClass();                                    // create Spider serial communicator

            // attach event listeners
            Spider1.SerialStatusChanged += Spider1_SerialStatusChanged;     //Function to be called when the connection status is changed
            Spider1.SerialError += Spider1_SerialError;                     //Function to be called when an error is detected
            Spider1.SerialPacketReceived += Spider1_SerialPacketReceived;   //Function to be called when a serial packet is received
            Spider1.GPIOStatusChanged += Spider1_GPIOStatusChanged;         //Function to be called when there is a GPIO status changed
            Spider1.GainSiChanged += Spider1_GainSiChanged;                 //Function to be called when the Si gain is changed
            Spider1.GainInGaAsChanged += Spider1_GainInGaAsChanged;         //Function to be called when the InGaAs gain is changed

            chart1.ChartAreas[0].AxisY.IsStartedFromZero = false;           //disable chart starting from zero

            comboBoxSamplingRate.SelectedIndex = 8;                         //sampling rate comboBox initialization (80kSPS is selected as default)
            updateTimeArray(comboBoxSamplingRate.SelectedIndex);            //initialize time_ms_array 
            comboBoxGainSi.SelectedIndex = 0;                               //initialize Si gain comboBox
            comboBoxGainInGaAs.SelectedIndex = 0;                           //initialize InGaAs gain comboBox 
            comboBoxAdcDigFilter.SelectedIndex = 0;                         //initialize ADC digital filter comboBox
            comboBoxYScalingMode.SelectedIndex = 1;                         //scaling mode = automatic     
            comboBoxTriggerOut.SelectedIndex = 0;                           //triggerOut = 0ff
        }

        private void Form1_Load(object sender, EventArgs e)                 //form load callback function
        {

        }

        private void buttonConnect_Click(object sender, EventArgs e)        //Connect button click callback
        {
            if(connectionStatus==0)                                         //if not connected
            {
                Spider1.OpenConnection(comboBoxPort.SelectedItem.ToString());   //open serial connection passing the COM port defined in the comboBox

                //enable GUI controls once Spider is connected
                groupBox2.Enabled = true;
                groupBox3.Enabled = true;
                groupBox4.Enabled = true;
                groupBox5.Enabled = true;

                Spider1.ReadStatus();                                   //read spider settings status to update the interface
            }
            else   
            {
                Spider1.StopContinuous();
                Spider1.CloseConnection();  //close serial connection

                //disable GUI controls when Spider is disconnected
                groupBox2.Enabled = false;
                groupBox3.Enabled = false;
                groupBox4.Enabled = false;
                groupBox5.Enabled = false;
            }

        }

        private void comboBoxPort_DropDown(object sender, System.EventArgs e)  //refresh the port name list when drop down the combobox
        {
            comboBoxPort.Items.Clear();                         //clear item list
            foreach (string s in SerialPort.GetPortNames())     //populate comboBox with all available COM ports
            {
                comboBoxPort.Items.Add(s);
            }

            if (comboBoxPort.Items.Count > 0) {comboBoxPort.SelectedIndex = 0;}     //if there is at least one element, select the first one
        }

        private void Spider1_SerialStatusChanged(object sender, EventArgs e)        //function called when the connection status is changed
        {
            if (Spider1.IsActive)                                                   //if IsActive is true = spider is connected
            {
                connectionStatus = 1;                                               //connection flag
                buttonConnect.BackColor = Color.Lime;                               //change connect button backcolor
                textBoxStatus.Text = "Connected";                                   //update Status textBox
            }
            else
            {
                connectionStatus = 0;                                               //connection flag
                buttonConnect.BackColor = Color.Transparent;                        //change connect button backcolor
                textBoxStatus.Text = "Not connected";                               //update Status textBox
            }
        }

        private void Spider1_SerialError(object sender, SerialErrorEventArgs e)     //function called when a serial error is detected
        {
            buttonConnect.BackColor = Color.Red;                                    //change connect button backcolor           

            string msg = "";                                                        
            switch (e.Error)                                                        //select case depending on the Error argument
            {
                case SpiderClass.SerialErrors.InvalidData:                          //error=invalid data
                    msg = "Invalid data received";
                    break;
                case SpiderClass.SerialErrors.TimeOut:                              //error=time out
                    msg = "Connection timed out";
                    break;
                case SpiderClass.SerialErrors.TransmissionError:                    //error=transmission error
                    msg = "Transmission error";
                    break;
            }
            Spider1.CloseConnection();                                              //close connection if an error is detected
            textBoxStatus.Text = msg;                                               //update Status textBox
            MessageBox.Show("Serial error", msg, MessageBoxButtons.OK);             //show error message
        }

        private void Spider1_SerialPacketReceived(object sender, SerialPacketReceivedEventArgs e)  //function called when a packet is received
        {
            plot_voltage_Si_array[0] = e.Voltage_Si_array[0];               //store first element of Si voltage array
            plot_voltage_InGaAs_array[0] = e.Voltage_InGaAs_array[0];       //store first element of InGaAs voltage array

            //if record data stream checkbox is checked, add the first samples and the GPIO status to the queue
            if (checkBoxRecord.Checked) { stream_data.Enqueue(plot_voltage_Si_array[0].ToString(CultureInfo.InvariantCulture) + "," + plot_voltage_InGaAs_array[0].ToString(CultureInfo.InvariantCulture) + "," + e.status_GPIO1.ToString() + "," + e.status_GPIO2.ToString() + "," + e.status_GPIO3.ToString()); }    //if record data stream checkbox is checked, add the 256 points Si and InGaAs voltage array to the queue (e.g. "1.23456,-1.23456")

            for (int i = 1; i < 256; i++)
            {
                plot_voltage_Si_array[i] = e.Voltage_Si_array[i];               //store Si voltage array
                plot_voltage_InGaAs_array[i] = e.Voltage_InGaAs_array[i];       //store InGaAs voltage array
                if (checkBoxRecord.Checked) { stream_data.Enqueue(plot_voltage_Si_array[i].ToString(CultureInfo.InvariantCulture) + "," + plot_voltage_InGaAs_array[i].ToString(CultureInfo.InvariantCulture));}    //if record data stream checkbox is checked, add the 256 points Si and InGaAs voltage array to the queue (e.g. "1.23456,-1.23456")
            }

            if(time_to_update>0)            //update chart if time_to_update flag is greater than 0
            {
                time_to_update = 0;         //reset flag

                if(checkBoxplotSi.Checked)   //if plot Si is checked
                {
                    chart1.Series[0].Points.DataBindXY(time_ms_array, plot_voltage_Si_array);   //plot voltage Si array
                    print_mean_rms_Si(e.Voltage_Si_array);                                         //compute mean and rms value, print everything to textBox
                }

                if (checkBoxplotInGaAs.Checked)   //if plot InGaAs is checked
                {
                    chart1.Series[1].Points.DataBindXY(time_ms_array, plot_voltage_InGaAs_array);//plot voltage InGaAs array
                    print_mean_rms_InGaAs(e.Voltage_InGaAs_array);                                      //compute mean and rms value, print everything to textBox
                }

                if (comboBoxYScalingMode.SelectedIndex > 0) { autoscaleChart(); }                //if scaling mode is Automatic -> autoscale chart
                else if (chart_max > chart_min)                                                  //otherwise (scaling mode is manual) if max limit is greater than min limit
                {
                    chart1.ChartAreas[0].AxisY.Maximum = chart_max;                              //set maximum limit of chart1
                    chart1.ChartAreas[0].AxisY.Minimum = chart_min;                              //set minimum limit of chart1
                }
            }
        }

        private void Spider1_GainSiChanged(object sender, GainChangedEventArgs e)            //function called when the Spider Si gain is changed (button pressed)
        {
            comboBoxGainSi.SelectedIndex = e.ReceivedData;                                      //update comboBox with the new gain value
        }

        private void Spider1_GainInGaAsChanged(object sender, GainChangedEventArgs e)        //function called when the Spider InGaAs gain is changed (button pressed)
        {
            comboBoxGainInGaAs.SelectedIndex = e.ReceivedData;                                  //update comboBox with the new gain value
        }

        private void Spider1_GPIOStatusChanged(object sender, GPIOChangedEventArgs e)        //function called when a digital input is changed
        {
            /*if((comboBoxTriggerOut.SelectedIndex==0) | (comboBoxSamplingRate.SelectedIndex < 4))      //if trigger out mode is disabled or refresh speed is not too high
            {
                textBoxIO1.Text = e.status_GPIO1.ToString();                             //write to the textbox the gpio1 status (converted to string)
            }*/
            textBoxIO1.Text = e.status_GPIO1.ToString();                             //write to the textbox the gpio1 status (converted to string)
            textBoxIO2.Text = e.status_GPIO2.ToString();                             //write to the textbox the gpio2 status (converted to string)
            textBoxIO3.Text = e.status_GPIO3.ToString();                             //write to the textbox the gpio3 status (converted to string)  

            comboBoxInOutIO1.SelectedIndex = e.dir_GPIO1;                            //update combobox with the real gpio1 direction of spider   
            comboBoxInOutIO2.SelectedIndex = e.dir_GPIO2;                            //update combobox with the real gpio2 direction of spider   
            comboBoxInOutIO3.SelectedIndex = e.dir_GPIO3;                            //update combobox with the real gpio3 direction of spider   
        }

        void print_mean_rms_Si(double[] array_voltage)                                             //compute mean and rms value of array_voltage, then print the results to textBox
        {   
            double mean = average(array_voltage);                                               //compute average value of array_voltage
            

            double rms = rootMeanSquare(array_voltage, mean);                                   //compute rms value of array_voltage

            ComputeAverageSi(rms);
            textBoxRMSSi.Text = Average_si.ToString("0.000E0");

            //textBoxRMSSi.Text = rms.ToString("0.000E0");                                          //convert the rms value to String with the scientic format
            textBoxMeanSi.Text = mean.ToString("0.00000E0");                                      //convert the mean value to String with the scientic format
            //textBoxENOBSi.Text = Math.Log(8.192 / rms, 2).ToString("0.0");                        //compute the ENOB (effective number of bits)
        }

        void print_mean_rms_InGaAs(double[] array_voltage)                                             //compute mean and rms value of array_voltage, then print the results to textBox
        {
            double mean = average(array_voltage);                                               //compute average value of array_voltage                             


            double rms = rootMeanSquare(array_voltage, mean);                                   //compute rms value of array_voltage

            ComputeAverageInGaAs(rms);
            textBoxRMSInGaAs.Text = Average_ingaas.ToString("0.000E0");

            //textBoxRMSInGaAs.Text = rms.ToString("0.000E0");                                          //convert the rms value to String with the scientic format
            textBoxMeanInGaAs.Text = mean.ToString("0.00000E0");                                      //convert the mean value to String with the scientic format
            //textBoxENOBInGaAs.Text = Math.Log(8.192 / rms, 2).ToString("0.0");                        //compute the ENOB (effective number of bits)
        }

        private static double rootMeanSquare(double[] x, double mean)                           //compute the rms value of x array subtracting the mean value
        {
            double sum = 0;
            for (int i = 0; i < x.Length; i++)
            {
                x[i] -= mean;                                                                   //subtract mean value from each element of the array
                sum += (x[i] * x[i]);                                                           //compute square value and add the result to "sum"
            }
            return Math.Sqrt(sum / x.Length);                                                   //compute the root square of the mean value
        }

        private static double average(double[] x)                                               //compute average value of x array
        {
            double sum = 0;
            for (int i = 0; i < x.Length; i++)                                                  //sum all the array elements
            {
                sum += x[i];                                  
            }
            return sum / x.Length;                                                              //return the sum divided by the number of elements in x
        }

        private void autoscaleChart() 
        {
            chart1.ChartAreas[0].AxisY.Maximum = Double.NaN;    // sets the Maximum to NaN
            chart1.ChartAreas[0].AxisY.Minimum = Double.NaN;    // sets the Minimum to NaN
            chart1.ChartAreas[0].RecalculateAxesScale();        // recalculates the Maximum and Minimum values, since they are set to NaN
            chart_max = chart1.ChartAreas[0].AxisY.Maximum;     // save maximum limit
            chart_min = chart1.ChartAreas[0].AxisY.Minimum;     // save minimum limit
            textBoxMax.Text = chart_max.ToString(CultureInfo.InvariantCulture);             // print max limit
            textBoxMin.Text = chart_min.ToString(CultureInfo.InvariantCulture);             // print min limit
        }

        private void updateTimeArray(int index)                 //recalculate time array
        {
            double timeScaleFactor = 1000 / (double)sampling_rate_combo[index];   //time step (in ms) from two consecutive samples  

            for (int i = 0; i < MAX_LENGTH; i++)                //compute the 256 values of time array
            {
                time_ms_array[i] = i * timeScaleFactor;                       
            }
        }

        private void comboBoxSamplingRate_SelectedIndexChanged(object sender, EventArgs e)
        {  
            updateTimeArray(comboBoxSamplingRate.SelectedIndex);            //recalculate time array
            Spider1.SetSamplingRate(comboBoxSamplingRate.SelectedIndex);    //set the sampling rate in Spider
            textBoxBWSi.Text = Spider1.ComputeEqBW(comboBoxGainSi.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
            textBoxBWInGaAs.Text = Spider1.ComputeEqBW(comboBoxGainInGaAs.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
        }

        private void buttonOneshot_Click(object sender, EventArgs e)
        {
            Spider1.OneShot();          //read only one packet of 256 values from Si and InGaAs channels
        }

        private void buttonStart_Click(object sender, EventArgs e)
        {
            stream_data.Clear();                        //clear stram data queue    
            stream_data.Enqueue("Si [V], InGaAs [V], GPIO1, GPIO2, GPIO3");    //add header text to queue
            Spider1.StartContinuous();                  //send a start acquisition command to Spider
        }

        private void buttonStop_Click(object sender, EventArgs e)
        {
            Spider1.StopContinuous();                   //send a stop acquisition command to Spider

            if (checkBoxRecord.Checked)                 //save queue to file if checkBox Record is checked
            {
                string date = DateTime.Now.ToString("yyyy_MM_dd_HH_mm_ss");     //save actual date and time as a string with the format "yyyy_MM_dd_HH_mm_ss"
                save_path = save_directory + "\\" + date + "\\";                //add date and time to the saving path
                Directory.CreateDirectory(save_path);                           //Create directory if it does not exist

                saveSettings(save_path, date);                                  //save GUI settings to file passing path and date

                if (!File.Exists(save_path + "\\stream.csv"))                   //if there is not a file with the same name
                {     
                    using (StreamWriter sw = File.CreateText(save_path + "\\stream.csv"))  // Create the stream.csv file to write to.
                    {
                        foreach (string data in stream_data)
                        {
                            sw.WriteLine(data);                                 //write every line present in the queue
                        }
                    }
                }
            }
        }

        private void buttonReadStatus_Click(object sender, EventArgs e)
        {
            Spider1.ReadStatus();           //send read status command to Spider
        }

        private void comboBoxGainSi_SelectedIndexChanged(object sender, EventArgs e)
        {
            Spider1.SetGainSi(comboBoxGainSi.SelectedIndex);        //set Spider Si gain
            textBoxBWSi.Text = Spider1.ComputeEqBW(comboBoxGainSi.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
        }

        private void comboBoxGainInGaAs_SelectedIndexChanged(object sender, EventArgs e)
        {
            Spider1.SetGainInGaAs(comboBoxGainInGaAs.SelectedIndex); //set Spider InGaAs gain   
            textBoxBWInGaAs.Text = Spider1.ComputeEqBW(comboBoxGainInGaAs.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
        }

        private void comboBoxInOutGPIO1_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 1;
            byte direction = (byte) comboBoxInOutIO1.SelectedIndex;
            Spider1.SetDirectionGPIO(gpio, direction);              //set direction (IN or OUT) of the gpio1
        }

        private void comboBoxInOutGPIO2_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 2;
            byte direction = (byte)comboBoxInOutIO2.SelectedIndex;
            Spider1.SetDirectionGPIO(gpio, direction);              //set direction (IN or OUT) of the gpio2
        }

        private void comboBoxInOutGPIO3_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 3;
            byte direction = (byte)comboBoxInOutIO3.SelectedIndex;
            Spider1.SetDirectionGPIO(gpio, direction);              //set direction (IN or OUT) of the gpio3
        }

        private void comboBoxStatusGPIO1_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 1;
            byte status = (byte) comboBoxStatusIO1.SelectedIndex;
            Spider1.SetStatusGPIO(gpio, status);                    //set status (0 or 1) of the gpio1
        }

        private void comboBoxStatusGPIO2_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 2;
            byte status = (byte)comboBoxStatusIO2.SelectedIndex;
            Spider1.SetStatusGPIO(gpio, status);                    //set status (0 or 1) of the gpio2
        }

        private void comboBoxStatusGPIO3_SelectedIndexChanged(object sender, EventArgs e)
        {
            byte gpio = 3;
            byte status = (byte)comboBoxStatusIO3.SelectedIndex;
            Spider1.SetStatusGPIO(gpio, status);                    //set status (0 or 1) of the gpio3
        }


        private void buttonFolder_Click(object sender, EventArgs e) //show dialog to select the saving folder
        {
            using (var fbd = new FolderBrowserDialog())
            {
                DialogResult result = fbd.ShowDialog();         //show dialog

                if (result == DialogResult.OK && !string.IsNullOrWhiteSpace(fbd.SelectedPath))  //if the saving folder is valid
                {
                    save_directory = fbd.SelectedPath;          //store saving folder   
                    textBoxPath.Text = save_directory;          //update textBox
                    buttonSaveData.Enabled = true;              //enable buttonSaveData 
                    checkBoxRecord.Enabled = true;              //enable checkBox for data recording
                }
            }
        }

        private void buttonSaveData_Click(object sender, EventArgs e)       //save settings and voltage values from one packet of 256 (both Si and InGaAs)
        {

            string date = DateTime.Now.ToString("yyyy_MM_dd_HH_mm_ss");     //save actual date and time as a string with the format "yyyy_MM_dd_HH_mm_ss"
            save_path = save_directory + "\\" + date + "\\";                //add date and time to the saving path
            Directory.CreateDirectory(save_path);                           //Create directory if it does not exist
                
            saveSettings(save_path, date);                                  //save GUI settings to file passing path and date

            using (StreamWriter outfile = new StreamWriter(save_path + "data.csv"))   //Create the data.csv file to write to. 
            {
                string content = "Time [ms], Si [V], InGaAs [V]\n";         //add header line to content variable (\n means new line)

                for (int i = 0; i < 256; i++)                               //for cycle along the 256 elements of the array
                {
                    content += time_ms_array[i].ToString("0.00000000E0", CultureInfo.InvariantCulture) + ",";                //add [i] element of time_ms_array (e.g. "1.23456E0,")
                    content += plot_voltage_Si_array[i].ToString("0.00000000E0", CultureInfo.InvariantCulture) + ",";        //add [i] element of voltage_si_array (e.g. "1.23456E0,")
                    content += plot_voltage_InGaAs_array[i].ToString("0.00000000E0", CultureInfo.InvariantCulture) + "\n";   //add [i] element of voltage_ingaas_array (e.g. "1.23456E0,")
                }

                outfile.Write(content);                                     //write content to data.csv file
            }
        }

        private void saveSettings(string path, string date)                 //save all settings of the GUI to the file settings.txt
        {
            string content;

            using (StreamWriter outfile = new StreamWriter(path + "settings.txt"))      //create settings.txt in saving path
            {
                content = "Date: " + date + "\n";                       //add e.g. "Date: yyyy_MM_dd_HH_mm_ss" to content with a newline command at the end
                content += "Note: " + textBoxNote.Text + "\n";
                content += "Port: " + comboBoxPort.SelectedItem.ToString() + "\n";
                content += "Sampling Rate: " + comboBoxSamplingRate.SelectedItem.ToString() + "\n";
                content += "Gain Si: " + comboBoxGainSi.SelectedItem.ToString() + "\n";
                content += "Gain InGaAs: " + comboBoxGainInGaAs.SelectedItem.ToString() + "\n";
                content += "GPIO0 Status: " + textBoxIO1.Text + "\n";
                content += "GPIO1 Status: " + textBoxIO2.Text + "\n";
                content += "GPIO2 Status: " + textBoxIO3.Text + "\n";
                content += "GPIO0 Direction: " + comboBoxInOutIO1.SelectedItem.ToString() + "\n";
                content += "GPIO1 Direction: " + comboBoxInOutIO2.SelectedItem.ToString() + "\n";
                content += "GPIO2 Direction: " + comboBoxInOutIO3.SelectedItem.ToString() + "\n";
                content += "Mean Si plot: " + textBoxMeanSi.Text + "\n";
                content += "RMS Si plot: " + textBoxRMSSi.Text + "\n";
                content += "Mean InGaAs plot: " + textBoxMeanInGaAs.Text + "\n";
                content += "RMS InGaAs plot: " + textBoxRMSInGaAs.Text + "\n";

                outfile.Write(content);     //write content to settings.txt file
            }
        }
        private void textBoxMax_TextChanged(object sender, EventArgs e)     //change maximum limit of chart 1
        {
            if(comboBoxYScalingMode.SelectedIndex == 0)                 //if manual scaling mode is selected
            {
                double value;
                bool success = double.TryParse(textBoxMax.Text, System.Globalization.NumberStyles.Any, System.Globalization.CultureInfo.InvariantCulture, out value);   //parse double from text ("." is decimal)
                if (success)                                            //if textBox contains a valid number
                {
                    if ((value - chart_min) < 0.000001) { chart_max = chart_min + 0.000001; }       //chart_max must be at least 0.000001 above chart_min, if this is not true set max limit = min limit + 0.000001
                    else if (value > 10) { chart_max = 10; }            //limit maximum value to 10V
                    else { chart_max = value; }                         //otherwise the value is in the acceptable range

                    chart1.ChartAreas[0].AxisY.Maximum = chart_max;     //set maximum limit
                }
            }
        }

        private void textBoxMin_TextChanged(object sender, EventArgs e)     //change minimum limit of chart 1
        {
            if (comboBoxYScalingMode.SelectedIndex == 0)            //if manual scaling mode is selected
            {
                double value;
                bool success = double.TryParse(textBoxMin.Text, System.Globalization.NumberStyles.Any, System.Globalization.CultureInfo.InvariantCulture, out value);   //parse double from text ("." is decimal)
                if (success)                                        //if textBox contains a valid number
                {
                    if ((chart_max - value) < 0.000001) { chart_min = chart_max - 0.000001; }    //chart_min must be at least 0.000001 below chart_max, if this is not true set min limit = max limit - 0.000001
                    else if (value < -10) { chart_min = -10; }      //limit minimum value to -10V
                    else { chart_min = value; }                     //otherwise the value is in the acceptable range

                    chart1.ChartAreas[0].AxisY.Minimum = chart_min; //set minimum limit
                }
            }
        }

        private void comboBoxYScalingMode_SelectedIndexChanged(object sender, EventArgs e)      //change automatic or manual scaling mode 
        {
            if(comboBoxYScalingMode.SelectedIndex == 0)              //0=manual scaling mode
            {
                textBoxMax.ReadOnly = false;                         //disable ReadOnly property of textBox, so it is editable
                textBoxMin.ReadOnly = false;                         //disable ReadOnly property of textBox, so it is editable
            }
            else                                                     //1=automatic scaling mode
            {
                textBoxMax.ReadOnly = true;                          //enable ReadOnly property of textBox, so it is not editable           
                textBoxMin.ReadOnly = true;                          //enable ReadOnly property of textBox, so it is not editable  
            }
        }

        private void timerUpdateChart_Tick(object sender, EventArgs e)  //the timer is used to limit the maximum refresh rate of chart
        {
            time_to_update = 1;                                         //set to 1 the timer flag, this occurs every 100ms (10Hz)
        }

        private void checkBoxplotSi_CheckedChanged(object sender, EventArgs e)
        {
            chart1.Series[0].Enabled = checkBoxplotSi.Checked;          //enable or disable the Si series on chart
        }

        private void checkBoxplotInGaAs_CheckedChanged(object sender, EventArgs e)
        {
            chart1.Series[1].Enabled = checkBoxplotInGaAs.Checked;      //enable or disable the InGaAs series on chart
        }

        private void comboBoxAdcDigFilter_SelectedIndexChanged(object sender, EventArgs e)
        {
            Spider1.SetAdcDigFilter(comboBoxAdcDigFilter.SelectedIndex);        //set Spider ADC digital filter
            textBoxBWSi.Text = Spider1.ComputeEqBW(comboBoxGainSi.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
            textBoxBWInGaAs.Text = Spider1.ComputeEqBW(comboBoxGainInGaAs.SelectedIndex, comboBoxSamplingRate.SelectedIndex, comboBoxAdcDigFilter.SelectedIndex).ToString();
        }

        private void button1_Click(object sender, EventArgs e)
        {
            Spider1.SetTriggerFunction(1, 1, 3, 32, 0);
        }

        private void comboBoxTriggerOut_SelectedIndexChanged(object sender, EventArgs e)
        {
            int index = comboBoxTriggerOut.SelectedIndex;
            if (index > 0) { comboBoxInOutIO1.SelectedIndex = 1; comboBoxInOutIO1.Enabled = false; comboBoxStatusIO1.Enabled = false; textBoxIO1.Enabled = false; }  //if Trigger out is enabled, set IO1 as an OUTPUT and disable manual controls
            else { comboBoxInOutIO1.Enabled = true; comboBoxStatusIO1.Enabled = true; textBoxIO1.Enabled = true; } //if Trigger out is disabled, enable manual controls
            Spider1.TriggerOutMode(index);
        }

        private Queue<double> samples_si = new Queue<double>();
        private int windowSize = 32;
        private double sampleAccumulator_si;
        public double Average_si { get; private set; }

        /// <summary>
        /// Computes a new windowed average each time a new sample arrives
        /// </summary>
        /// <param name="newSample"></param>
        public void ComputeAverageSi(double newSample)
        {
            sampleAccumulator_si += newSample;
            samples_si.Enqueue(newSample);

            if (samples_si.Count > windowSize)
            {
                sampleAccumulator_si -= samples_si.Dequeue();
            }

            Average_si = sampleAccumulator_si / samples_si.Count;
        }

        private Queue<double> samples_ingaas = new Queue<double>();
        private double sampleAccumulator_ingaas;
        public double Average_ingaas { get; private set; }

        /// <summary>
        /// Computes a new windowed average each time a new sample arrives
        /// </summary>
        /// <param name="newSample"></param>
        public void ComputeAverageInGaAs(double newSample)
        {
            sampleAccumulator_ingaas += newSample;
            samples_ingaas.Enqueue(newSample);

            if (samples_ingaas.Count > windowSize)
            {
                sampleAccumulator_ingaas -= samples_ingaas.Dequeue();
            }

            Average_ingaas = sampleAccumulator_ingaas / samples_ingaas.Count;
        }
    }
}
